/*******************************************************************************
********************************************************************************
** COPYRIGHT:    (c) 2013 Rohde & Schwarz
** MODULE:       CSCPIHelper_PR100.cpp
** LANGUAGE:     C/C++
** AUTHOR:       
** ABSTRACT:     Helper class to encapsulate SCPI access for PR100
** PREMISES:
** REMARKS:
** HISTORY:
********************************************************************************/

/* INCLUDE FILES ***************************************************************/
#include "CSCPIHelper_PR100.h"

/* CDevice Implementation ***************************************************************/

CSCPIHelper::CSCPIHelper(std::string remoteIP, unsigned short remotePort) :
    m_sIP           (remoteIP),
    m_nIP           (0),
    m_nPort         (remotePort),
    m_Socket        (SOCKET_ERROR),
    m_nOptions      (0),
    m_sDeviceName   (""),
    m_nSN           (0),
    m_sFWVer        (""),
    m_bDebugOutput  (false)
{
}


bool CSCPIHelper::ResolveIP()
{
    /* Determine receiver device IP address */
    m_nIP = inet_addr(m_sIP.c_str());
    if (m_nIP == INADDR_NONE)
    {
        hostent* phost = gethostbyname(m_sIP.c_str());
        if (phost && phost->h_addr_list[0] != NULL)
        {
            m_nIP = *((unsigned*) phost->h_addr_list[0]);
        }
        else
        {
            if(m_bDebugOutput) printf("ERROR: Cannot resolve host name %s.\n", m_sIP);
            return false;
        }
    }

    return true;
}

bool CSCPIHelper::Connect()
{
    if(ResolveIP() == false)
    {
        return false;
    }

    /* Create a receiver control TCP socket */
    m_Socket = socket(AF_INET, SOCK_STREAM, 0);
    if (m_Socket == SOCKET_ERROR)
    {
        if(m_bDebugOutput) printf("ERROR: Unable to create SCPI socket.\n");
        goto err_cleanup;
    }

    int sopt = 1;
    if ( setsockopt( m_Socket, IPPROTO_TCP, TCP_NODELAY, (char *)&sopt, sizeof( sopt ) ) == SOCKET_ERROR )
    {
        if(m_bDebugOutput) printf("ERROR: setsockopt (TCP_NODELAY, %d) failed - errno %d\n", sopt, WSAGetLastError());
        goto err_cleanup;
    }

    /* Create socket address structure for INET address family */
    struct sockaddr_in addrDevice;
    memset(&addrDevice, 0, sizeof(addrDevice));
    addrDevice.sin_family      = AF_INET;
    addrDevice.sin_addr.s_addr = m_nIP;
    addrDevice.sin_port        = htons(m_nPort);

    /* connect socket to receiver device */
    int err = connect(m_Socket, (struct sockaddr *)&addrDevice, sizeof(addrDevice));
    if (err != 0)
    {
        if(m_bDebugOutput) printf("ERROR: connecting to %s [SCPI port = %d], err = %d\n", m_sIP.c_str(), m_nPort, WSAGetLastError());
        goto err_cleanup;
    }

    /* Get the device properties */
    if( (GetDeviceIdentify() == false) || (GetDeviceOptions(m_nOptions) == false) )
    {
        if(m_bDebugOutput) printf("ERROR: Unable to read device properties!\n");
        goto err_cleanup;
    }

    return true;

err_cleanup:
    if(m_Socket != SOCKET_ERROR)
    {
        closesocket(m_Socket);
    }
    return false;

}

bool CSCPIHelper::Disconnect()
{
    if(m_Socket != SOCKET_ERROR)
    {
        closesocket(m_Socket);
    }

    return true;
}

bool CSCPIHelper::GetDeviceIdentify(void)
{
    char cBuffer[READ_SIZE_BYTES];
    char *pBuffer = cBuffer;
    int  len = 0;
    int  nResponseErr = 0;

    SendCmd("*IDN?\n");
    len = sizeof(cBuffer);
    nResponseErr = GetResponseWithTimeout(cBuffer, len, READ_TIMEOUT_MS);
    if( len < 0 ) len = 0; 
    cBuffer[ len ] = '\0';

    if(nResponseErr == 0)
    {
        //Expected format: ROHDE&SCHWARZ, <name>, <sn>, <fw version>
        char * token      = NULL;
        char * next_token = NULL;
        int field = 0;

        token = CSCPIHelper::Tokenise (cBuffer,",", &next_token);

        while (token != NULL)
        {
            token = CSCPIHelper::TrimBefore(token);

            switch(field)
            {
            case 0:
                // Company name should be "ROHDE&SCHWARZ"
                // Skip this field :)
                break;

            case 1:
                if ( (strcmp(token,"PR100")  == 0) ||
                     (strcmp(token,"EM100")  == 0) ||
                     (strcmp(token,"DDF007") == 0) 
                    )
                {
                    m_sDeviceName = token;
                }
                else
                {
                    if(m_bDebugOutput) printf("ERROR: Unsupported device\n");
                    goto err_cleanup;
                }
                break;

            case 2:
                //Should be serial number
                m_nSN = atoi(token);
                break;

            case 3:
                m_sFWVer = token;
                break;

            default:
                //Should not reach here (maybe corrupted reply?)
                break;
            }

            field++;
            token = CSCPIHelper::Tokenise (NULL, ",", &next_token);
        }

        if(GetLastCmdErr() != 0)
        {
            if(m_bDebugOutput) printf("ERROR: *IDN? error\n");
            goto err_cleanup;
        }
    }
    else
    {
        if(m_bDebugOutput) printf("ERROR: Error getting response\n");
        goto err_cleanup;
    }

    if(m_bDebugOutput) printf("Device=%s\n", m_sDeviceName.c_str());
    return true;

err_cleanup:
    return false;
}

bool CSCPIHelper::GetDeviceOptions(unsigned int& options)
{
    char cBuffer[READ_SIZE_BYTES];
    int len = 0;
    int nResponseErr = 0;
    options = 0;

    SendCmd("*OPT?\n");
    len = sizeof(cBuffer);
    nResponseErr = GetResponseWithTimeout(cBuffer, len, READ_TIMEOUT_MS);
    if( len < 0 ) len = 0; 
    cBuffer[ len ] = '\0';

    if(nResponseErr == 0)
    {
        if(GetLastCmdErr() != 0)
        {
            if(m_bDebugOutput) printf("*OPT? err\n");
            return false;
        }
        else
        {
            // Parse the returned result
            char * token      = NULL;
            char * next_token = NULL;

            token = Tokenise (cBuffer,",", &next_token);

            while (token != NULL)
            {
                if(token[0] != '\0')
                {
                    if      (strcmp(token,"PS") == 0) options |= OPTION_PS;
                    else if (strcmp(token,"IR") == 0) options |= OPTION_IR;
                    else if (strcmp(token,"RC") == 0) options |= OPTION_RC;
                    else if (strcmp(token,"ET") == 0) options |= OPTION_ET;
                    else if (strcmp(token,"FS") == 0) options |= OPTION_FS;
                    else if (strcmp(token,"FP") == 0) options |= OPTION_FP;
                    else if (strcmp(token,"FE") == 0) options |= OPTION_FE;
                    else if (strcmp(token,"GP") == 0) options |= OPTION_GP;
                    else if (strcmp(token,"DF") == 0) options |= OPTION_DF;

                }

                token = Tokenise (NULL, ",", &next_token);
            }

            return true;
        }
    }
    else
    {
        return false;
    }
}

int CSCPIHelper::SendCmd(const char *pBuffer)
{
    if(pBuffer == NULL)
    {
        return -1;
    }

    unsigned int nLen = 0;
    if(m_bDebugOutput) printf("SendCmd: %s", pBuffer);

    nLen = send(m_Socket, pBuffer, strlen(pBuffer), 0);
    if (nLen != strlen(pBuffer))
    {
        if(m_bDebugOutput) printf("Error writing to socket. Len = %d\n", nLen);
        return -1;
    }
    else
    {
        return 0;
    }
}

int CSCPIHelper::GetBinaryResponseWithTimeout( char *pBuffer, int& length, unsigned int msTimeOut )
{
    fd_set    fds;
    int       selectResult;
    int       nErr = 0;
    int       received = 0;
    struct    timeval tv;

    if( (pBuffer == NULL) || (length == 0))
    {
        nErr = -3;
        goto cleanup;
    }

    // set up the file descriptor set
    FD_ZERO(&fds);
    #pragma warning( disable : 4127 )
    FD_SET(m_Socket, &fds);
    #pragma warning( default : 4127 )

    // set up the struct timeval for the timeout
    tv.tv_sec = msTimeOut / 1000;
    tv.tv_usec = 0;

    do
    {
        // wait until timeout or target data requested received.
        selectResult = select((int)m_Socket, &fds, NULL, NULL, &tv);
        if (selectResult == 0) 
        {
            if(m_bDebugOutput) printf("Timed out\n");
            nErr = -1;
            goto cleanup;
        }
        else if (selectResult == SOCKET_ERROR)
        {
            if(m_bDebugOutput) printf("Socket error\n");
            nErr = -2;
            goto cleanup;
        }

        // data must be here, so do a normal receive
        int nLen = recv(m_Socket, pBuffer, length - received, 0);
        pBuffer  += nLen;
        received += nLen;
    }
    while (received < length);

    length = received;
    return 0;

cleanup:
    length = 0;
    return nErr;

}

int CSCPIHelper::GetResponseWithTimeout( char *pBuffer, int& length, unsigned int msTimeOut )
{
    fd_set    fds;
    int       selectResult;
    int       nErr = 0;
    int       received = 0;
    struct    timeval tv;
    bool      bResponseTerminated = false;

    if( (pBuffer == NULL) || (length == 0) )
    {
        nErr = -3;
        goto cleanup;
    }

    // set up the file descriptor set
    FD_ZERO(&fds);
#pragma warning( disable : 4127 )
    FD_SET(m_Socket, &fds);
#pragma warning( default : 4127 )

    // set up the struct timeval for the timeout
    tv.tv_sec = msTimeOut / 1000;
    tv.tv_usec = 0;

    // init string charBuf
    char c = '\0';
    do
    {
        // wait until timeout or data received.
        selectResult = select((int)m_Socket, &fds, NULL, NULL, &tv);
        if (selectResult == 0) 
        {
            if(m_bDebugOutput) printf("Timed out\n");
            nErr = -1;
            goto cleanup;
        }
        else if (selectResult == SOCKET_ERROR)
        {
            if(m_bDebugOutput) printf("Socket error\n");
            nErr = -2;
            goto cleanup;
        }

        // data must be here, so do a normal receive (one byte at a time)
        recv(m_Socket, &c, 1, 0);

        // ignore <CR> and <LF>, all other data put into buffer
        if (c == '\n') 
        {
            bResponseTerminated = true;
        }
        else if (c == '\r') 
        {
            //Discard \r
        }
        else
        {
            *pBuffer = c;
            pBuffer++;
            received++;
        }
    }
    while ( (bResponseTerminated == false) && (received < length) );

    //String terminator
    *pBuffer = '\0';

    length = received;
    if(bResponseTerminated)
    {
        return 0;
    }
    else
    {
        return -4;
    }

cleanup:
    length = 0;
    return nErr;
}

int CSCPIHelper::GetBlockData( CallbackFunc callback, void * pPrivate )
{
	/* create buffer for receiving */
	char pRxBuf[READ_BLOCKDATA_SIZE_BYTES];
    int       nErr            = 0;
    int       nResponseErr    = 0;
    int       len             = 0;

    BLOCKDATA_T info;
    memset(&info, 0, sizeof(BLOCKDATA_T));

    /* read the "header" part of binary data to get the length

       number of digits: if you receive the data, than 
	   the second character is the number of digits for the size 
	   of the file, for example: #530984...... 
	   here 5 is the number of digits and then 30984 (5 characters) 
	   is the size of the file. This means that you have to store 
	   your file after the file size, in this case in the 7th 
	   position of the buffer 
    */

    len = 2;
    nResponseErr = GetBinaryResponseWithTimeout(pRxBuf, len, READ_TIMEOUT_MS);
    if(nResponseErr == 0)
    {
        if((len == 2) && (pRxBuf[0] == '#'))
        {
            // number of digits to read next
            int nchars = pRxBuf[1] - '0'; 

            len = nchars;
            nResponseErr = GetBinaryResponseWithTimeout(pRxBuf, len, READ_TIMEOUT_MS);
            if( (nResponseErr == 0) && (len == nchars) )
            {
                pRxBuf[len] = '\0';
                info.totalDataSize = atoi(pRxBuf);
            }
            else
            {
                nErr = -2;
                if(m_bDebugOutput) printf("ERROR: corrupted reply\n");
                goto err_cleanup;
            }
        }
        else
        {
            nErr = -2;
            if(m_bDebugOutput) printf("ERROR: corrupted reply\n");
            goto err_cleanup;
        }

    }
    else
    {
        nErr = -1;
        if(m_bDebugOutput) printf("ERROR: Error reading response %d\n", nResponseErr);
        goto err_cleanup;
    }

	/* Loop reading bytes into the file */
	while(info.offset < info.totalDataSize)
	{
        int remain_len = info.totalDataSize - info.offset;
        len = ((remain_len > sizeof(pRxBuf)) ? sizeof(pRxBuf) : remain_len);

        nResponseErr = GetBinaryResponseWithTimeout(pRxBuf, len, READ_TIMEOUT_MS);
        if(nResponseErr == 0)
        {
            info.pBufferData = pRxBuf;
            info.bufferSize  = len;
            info.status      = BLOCKDATA_STATUS_DATA;
            callback(pPrivate, info);

            info.offset += info.bufferSize; //Update the offset AFTER the callback
        }
        else
        {
            nErr = -1;
            if(m_bDebugOutput) printf("ERROR: Error reading response %d\n", nResponseErr);
            goto err_cleanup;
        }
	}

    // Read out the \n\r characters and discard them
    len = 2;
    nResponseErr = GetBinaryResponseWithTimeout(pRxBuf, len, READ_TIMEOUT_MS);
    if(nResponseErr == 0)
    {
        if( (len == 2) && (pRxBuf[0] != '\r') && (pRxBuf[1] != '\n') )
        {
            if(m_bDebugOutput) printf("WARNING: Incorrect terminating characters!\n");
        }
    }
    else
    {
        if(m_bDebugOutput) printf("WARNING: timeout reading terminating characters!\n");
    }


err_cleanup: //Common cleanup section

    info.status = (nErr == 0) ? BLOCKDATA_STATUS_DONE : BLOCKDATA_STATUS_FAIL;
    info.pBufferData = NULL;
    info.bufferSize  = 0;
    callback(pPrivate, info);

    return nErr;
}

int CSCPIHelper::GetLastCmdErr()
{
    int  len = 0;
    int  nResponseErr = 0;
    int  nNumErrs = 0;
    int  nLastErr = 0;

    char cBuffer[READ_SIZE_BYTES];
    SendCmd("*WAI;:SYST:ERR:CODE?\n");

    len = sizeof(cBuffer);
    nResponseErr = GetResponseWithTimeout(cBuffer, len, READ_TIMEOUT_MS);
    if(nResponseErr != 0)
    {
        if(m_bDebugOutput) printf("Error getting response\n");
        return nResponseErr;
    }

    if ( len < 0 ) len = 0; 
    cBuffer[ len ] = '\0';

    // Parse the returned result
    char * token      = NULL;
    char * next_token = NULL;

    token = CSCPIHelper::Tokenise (cBuffer,",", &next_token);

    while (token != NULL)
    {
        if(token[0] != '\0')
        {
            int nSCPIErr = atoi(token);
            if(nSCPIErr != 0)
            {
                nNumErrs++;
                nLastErr = nSCPIErr;
                if(m_bDebugOutput) printf("Found err = %d\n", nSCPIErr);
            }
        }

        token = CSCPIHelper::Tokenise (NULL, ",", &next_token);
    }

    if(m_bDebugOutput) printf("GetLastCmdErr: NumErr = %d, lastErr = %d\n", nNumErrs, nLastErr);

    return nLastErr;
}

// Custom implementation to return empty string.
char * CSCPIHelper::Tokenise(char *str, const char *delim, char **nextp)
{
    char *ret;

    if (str == NULL)
    {
        str = *nextp;
    }
/*
    // comment out this line so that consecutive delimiters are not considered as one.
    str += strspn(str, delim);
*/

    if (*str == '\0')
    {
        return NULL;
    }

    ret = str;

    str += strcspn(str, delim);

    if (*str)
    {
        *str = '\0';
        str++;
    }

    *nextp = str;

    return ret;
}

//Remove whitespaces before the string.
char * CSCPIHelper::TrimBefore(char *str)
{
    while(str && (*str == ' ') )
    {
        str++;
    }
    return str;
}